<?php
session_start();
require_once dirname(__DIR__, 2) . DIRECTORY_SEPARATOR . 'config' . DIRECTORY_SEPARATOR . 'config.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: ' . getUrlPath('pages/login.php'));
    exit();
}

// Get database connection
$pdo = getDBConnection();

// Get products for dropdown
$products = getRows("SELECT * FROM inventory WHERE quantity > 0 ORDER BY product_name");

$errors = [];
$success = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $patient_name = $_POST['patient_name'] ?? '';
    $dispensing_date = $_POST['dispensing_date'] ?? '';
    $prescription_number = $_POST['prescription_number'] ?? '';
    $products_data = $_POST['products'] ?? [];

    if (empty($patient_name)) {
        $errors[] = "Please enter patient name";
    }

    if (empty($dispensing_date)) {
        $errors[] = "Please select a dispensing date";
    }

    if (empty($products_data)) {
        $errors[] = "Please add at least one product";
    }

    if (empty($errors)) {
        try {
            $pdo->beginTransaction();

            // Insert dispensing record
            $sql = "INSERT INTO dispensing_records (patient_name, dispensing_date, prescription_number, dispensed_by) 
                    VALUES (?, ?, ?, ?)";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([$patient_name, $dispensing_date, $prescription_number, $_SESSION['user_id']]);
            $dispensing_id = $pdo->lastInsertId();

            // Insert dispensing items and update inventory
            $sql = "INSERT INTO dispensing_items (dispensing_record_id, product_id, quantity) VALUES (?, ?, ?)";
            $stmt = $pdo->prepare($sql);

            foreach ($products_data as $product) {
                // Check if enough quantity is available
                $check_sql = "SELECT quantity FROM inventory WHERE id = ?";
                $check_stmt = $pdo->prepare($check_sql);
                $check_stmt->execute([$product['product_id']]);
                $current_quantity = $check_stmt->fetchColumn();

                if ($current_quantity < $product['quantity']) {
                    throw new Exception("Insufficient quantity for product ID: " . $product['product_id']);
                }

                // Insert dispensing item
                $stmt->execute([
                    $dispensing_id,
                    $product['product_id'],
                    $product['quantity']
                ]);

                // Update inventory quantity
                $sql = "UPDATE inventory SET quantity = quantity - ? WHERE id = ?";
                $stmt = $pdo->prepare($sql);
                $stmt->execute([$product['quantity'], $product['product_id']]);
            }

            $pdo->commit();
            $success = true;
        } catch (Exception $e) {
            $pdo->rollBack();
            $errors[] = "Error processing dispensing: " . $e->getMessage();
        }
    }
}

require_once getIncludePath('includes/header.php');
?>

<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header">
                    <h3 class="card-title">Record Dispensing</h3>
                </div>
                <div class="card-body">
                    <?php if (!empty($errors)): ?>
                        <div class="alert alert-danger">
                            <ul class="mb-0">
                                <?php foreach ($errors as $error): ?>
                                    <li><?php echo htmlspecialchars($error); ?></li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                    <?php endif; ?>

                    <?php if ($success): ?>
                        <div class="alert alert-success">
                            Dispensing recorded successfully!
                        </div>
                    <?php endif; ?>

                    <form method="POST" id="dispensingForm">
                        <div class="row mb-4">
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label for="patient_name">Patient Name</label>
                                    <input type="text" class="form-control" id="patient_name" name="patient_name" required>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label for="dispensing_date">Dispensing Date</label>
                                    <input type="date" class="form-control" id="dispensing_date" name="dispensing_date" 
                                           value="<?php echo date('Y-m-d'); ?>" required>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label for="prescription_number">Prescription Number (Optional)</label>
                                    <input type="text" class="form-control" id="prescription_number" name="prescription_number">
                                </div>
                            </div>
                        </div>

                        <div id="productsContainer">
                            <div class="product-item mb-3">
                                <div class="row">
                                    <div class="col-md-5">
                                        <div class="form-group">
                                            <label>Product</label>
                                            <select class="form-control product-select" name="products[0][product_id]" required>
                                                <option value="">Select Product</option>
                                                <?php foreach ($products as $product): ?>
                                                    <option value="<?php echo $product['id']; ?>">
                                                        <?php echo htmlspecialchars($product['product_name']); ?> 
                                                        (Stock: <?php echo $product['quantity']; ?>)
                                                    </option>
                                                <?php endforeach; ?>
                                            </select>
                                        </div>
                                    </div>
                                    <div class="col-md-3">
                                        <div class="form-group">
                                            <label>Quantity</label>
                                            <input type="number" class="form-control" name="products[0][quantity]" 
                                                   min="1" required>
                                        </div>
                                    </div>
                                    <div class="col-md-4">
                                        <div class="form-group">
                                            <label>&nbsp;</label>
                                            <button type="button" class="btn btn-danger btn-block remove-product" style="display: none;">
                                                Remove
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-12">
                                <button type="button" class="btn btn-secondary" id="addProduct">Add Product</button>
                                <button type="submit" class="btn btn-primary">Save Dispensing</button>
                                <button type="reset" class="btn btn-default">Clear Form</button>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const productsContainer = document.getElementById('productsContainer');
    const addProductBtn = document.getElementById('addProduct');
    let productCount = 1;

    addProductBtn.addEventListener('click', function() {
        const template = document.querySelector('.product-item').cloneNode(true);
        
        // Update names and IDs
        template.querySelectorAll('select, input').forEach(input => {
            input.name = input.name.replace('[0]', `[${productCount}]`);
            input.value = '';
        });

        // Show remove button
        template.querySelector('.remove-product').style.display = 'block';

        productsContainer.appendChild(template);
        productCount++;
    });

    // Remove product
    productsContainer.addEventListener('click', function(e) {
        if (e.target.classList.contains('remove-product')) {
            e.target.closest('.product-item').remove();
        }
    });
});
</script>

<?php require_once getIncludePath('includes/footer.php'); ?> 