<?php
session_start();
require_once dirname(__DIR__, 2) . DIRECTORY_SEPARATOR . 'config' . DIRECTORY_SEPARATOR . 'config.php';
require_once getIncludePath('includes/header.php');

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: ' . getUrlPath('pages/login.php'));
    exit();
}

// Get database connection
$pdo = getDBConnection();

// Get all products for dropdown
$products = getRows("SELECT id, product_name, quantity, unit_of_measure FROM inventory ORDER BY product_name");

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $product_id = $_POST['product_id'];
    $quantity_change = $_POST['quantity_change'];
    $reason = $_POST['reason'];
    $adjustment_date = $_POST['adjustment_date'];
    $notes = $_POST['notes'];
    $staff_id = $_SESSION['user_id'];

    $errors = [];

    // Validate required fields
    if (empty($product_id)) {
        $errors[] = "Product selection is required";
    }
    if (empty($quantity_change)) {
        $errors[] = "Quantity change is required";
    }
    if (empty($reason)) {
        $errors[] = "Reason for adjustment is required";
    }
    if (empty($adjustment_date)) {
        $errors[] = "Adjustment date is required";
    }

    // Validate quantity change
    if (!is_numeric($quantity_change)) {
        $errors[] = "Quantity change must be a number";
    }

    if (empty($errors)) {
        try {
            $pdo->beginTransaction();

            // Insert adjustment record
            $adjustment_data = [
                'product_id' => $product_id,
                'quantity_change' => $quantity_change,
                'reason' => $reason,
                'adjustment_date' => $adjustment_date,
                'staff_id' => $staff_id,
                'notes' => $notes
            ];
            
            if (insertData('inventory_adjustments', $adjustment_data)) {
                // Update inventory quantity
                $update_query = "UPDATE inventory SET quantity = quantity + ? WHERE id = ?";
                $pdo->prepare($update_query)->execute([$quantity_change, $product_id]);
                
                $pdo->commit();
                $_SESSION['success_message'] = "Inventory adjustment recorded successfully.";
                header('Location: ' . getUrlPath('pages/inventory/list.php'));
                exit();
            } else {
                throw new Exception("Failed to record adjustment");
            }
        } catch (Exception $e) {
            $pdo->rollBack();
            $errors[] = "Error recording adjustment: " . $e->getMessage();
        }
    }
}
?>

<div class="container-fluid">
    <div class="row justify-content-center">
        <div class="col-md-8">
            <div class="card">
                <div class="card-header">
                    <h3 class="card-title mb-0">Record Inventory Adjustment</h3>
                </div>
                <div class="card-body">
                    <?php if (!empty($errors)): ?>
                        <div class="alert alert-danger">
                            <ul class="mb-0">
                                <?php foreach ($errors as $error): ?>
                                    <li><?php echo $error; ?></li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                    <?php endif; ?>

                    <form method="POST">
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label for="product_id" class="form-label">Product*</label>
                                <select class="form-select" id="product_id" name="product_id" required>
                                    <option value="">Select Product</option>
                                    <?php foreach ($products as $product): ?>
                                        <option value="<?php echo $product['id']; ?>" 
                                                data-current-quantity="<?php echo $product['quantity']; ?>"
                                                data-unit="<?php echo htmlspecialchars($product['unit_of_measure']); ?>">
                                            <?php echo htmlspecialchars($product['product_name']); ?> 
                                            (Current: <?php echo $product['quantity'] . ' ' . htmlspecialchars($product['unit_of_measure']); ?>)
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label for="quantity_change" class="form-label">Quantity Change*</label>
                                <div class="input-group">
                                    <input type="number" class="form-control" id="quantity_change" name="quantity_change" required>
                                    <span class="input-group-text" id="unit_of_measure">units</span>
                                </div>
                                <small class="text-muted">Use positive numbers for additions, negative for reductions</small>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label for="reason" class="form-label">Reason for Adjustment*</label>
                                <select class="form-select" id="reason" name="reason" required>
                                    <option value="">Select Reason</option>
                                    <option value="spoilage">Spoilage</option>
                                    <option value="damage">Damage</option>
                                    <option value="cycle_count">Cycle Count</option>
                                    <option value="theft">Theft</option>
                                    <option value="expiration">Expiration</option>
                                    <option value="other">Other</option>
                                </select>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label for="adjustment_date" class="form-label">Adjustment Date*</label>
                                <input type="date" class="form-control" id="adjustment_date" name="adjustment_date" 
                                       value="<?php echo date('Y-m-d'); ?>" required>
                            </div>
                        </div>

                        <div class="mb-3">
                            <label for="notes" class="form-label">Notes</label>
                            <textarea class="form-control" id="notes" name="notes" rows="3"></textarea>
                        </div>

                        <div class="d-grid gap-2">
                            <button type="submit" class="btn btn-primary">Record Adjustment</button>
                            <a href="<?php echo getUrlPath('pages/inventory/list.php'); ?>" class="btn btn-secondary">Cancel</a>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
document.getElementById('product_id').addEventListener('change', function() {
    const selectedOption = this.options[this.selectedIndex];
    const unitOfMeasure = selectedOption.dataset.unit || 'units';
    document.getElementById('unit_of_measure').textContent = unitOfMeasure;
});
</script>

<?php require_once getIncludePath('includes/footer.php'); ?> 