<?php
session_start();
require_once dirname(__DIR__, 2) . DIRECTORY_SEPARATOR . 'config' . DIRECTORY_SEPARATOR . 'config.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: ' . getUrlPath('pages/login.php'));
    exit();
}

// Get database connection
$pdo = getDBConnection();

// Check if ID is provided
if (!isset($_GET['id'])) {
    header('Location: ' . getUrlPath('pages/inventory/list.php'));
    exit();
}

$id = $_GET['id'];

// Get item details
$item = getRow("SELECT * FROM inventory WHERE id = ?", [$id]);

if (!$item) {
    $_SESSION['error_message'] = "Item not found.";
    header('Location: ' . getUrlPath('pages/inventory/list.php'));
    exit();
}

// Get suppliers for dropdown
$suppliers = getRows("SELECT * FROM suppliers ORDER BY name");

// Get current supplier ID
$current_supplier = getRow("SELECT id FROM suppliers WHERE name = ?", [$item['manufacturer']]);
$current_supplier_id = $current_supplier ? $current_supplier['id'] : '';

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $product_name = $_POST['product_name'];
    $generic_name = $_POST['generic_name'];
    $ndc = $_POST['ndc'];
    $sku = $_POST['sku'];
    $dosage_strength = $_POST['dosage_strength'];
    $form = $_POST['form'];
    $supplier_id = $_POST['supplier_id'];
    $unit_of_measure = $_POST['unit_of_measure'];
    $category = $_POST['category'];
    $storage_requirements = $_POST['storage_requirements'];
    $quantity = $_POST['quantity'];
    $reorder_point = $_POST['reorder_point'];
    $max_stock_level = $_POST['max_stock_level'];
    $expiry_date = $_POST['expiry_date'];
    $lot_number = $_POST['lot_number'];
    $unit_price = $_POST['unit_price'];
    $description = $_POST['description'];

    $errors = [];

    // Validate required fields
    if (empty($product_name)) {
        $errors[] = "Product name is required";
    }
    if (empty($sku)) {
        $errors[] = "SKU is required";
    }
    if (empty($supplier_id)) {
        $errors[] = "Please select a supplier";
    }

    // Validate numeric fields
    if (!is_numeric($quantity) || $quantity < 0) {
        $errors[] = "Quantity must be a non-negative number";
    }
    if (!is_numeric($unit_price) || $unit_price <= 0) {
        $errors[] = "Unit price must be greater than 0";
    }
    if (!is_numeric($reorder_point) || $reorder_point < 0) {
        $errors[] = "Reorder point must be a non-negative number";
    }
    if (!empty($max_stock_level) && (!is_numeric($max_stock_level) || $max_stock_level < $reorder_point)) {
        $errors[] = "Maximum stock level must be greater than reorder point";
    }

    // Validate expiry date if provided
    if (!empty($expiry_date)) {
        $expiry_timestamp = strtotime($expiry_date);
        if ($expiry_timestamp === false) {
            $errors[] = "Invalid expiry date format";
        }
    }

    // Handle image upload
    $product_image = $item['product_image'];
    if (isset($_FILES['product_image']) && $_FILES['product_image']['error'] == 0) {
        $allowed = ['jpg', 'jpeg', 'png', 'gif'];
        $filename = $_FILES['product_image']['name'];
        $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
        
        if (in_array($ext, $allowed)) {
            $new_filename = uniqid() . '.' . $ext;
            $upload_path = dirname(__DIR__, 2) . DIRECTORY_SEPARATOR . 'uploads' . DIRECTORY_SEPARATOR . 'products' . DIRECTORY_SEPARATOR . $new_filename;
            
            if (!file_exists(dirname(__DIR__, 2) . DIRECTORY_SEPARATOR . 'uploads' . DIRECTORY_SEPARATOR . 'products')) {
                mkdir(dirname(__DIR__, 2) . DIRECTORY_SEPARATOR . 'uploads' . DIRECTORY_SEPARATOR . 'products', 0777, true);
            }
            
            if (move_uploaded_file($_FILES['product_image']['tmp_name'], $upload_path)) {
                // Delete old image if exists
                if ($product_image && file_exists(dirname(__DIR__, 2) . DIRECTORY_SEPARATOR . $product_image)) {
                    unlink(dirname(__DIR__, 2) . DIRECTORY_SEPARATOR . $product_image);
                }
                $product_image = 'uploads/products/' . $new_filename;
            } else {
                $errors[] = "Failed to upload image";
            }
        } else {
            $errors[] = "Invalid image format. Allowed formats: JPG, JPEG, PNG, GIF";
        }
    }

    if (empty($errors)) {
        try {
            $pdo->beginTransaction();

            // Get supplier name
            $supplier = getRow("SELECT name FROM suppliers WHERE id = ?", [$supplier_id]);
            if (!$supplier) {
                throw new Exception("Selected supplier not found");
            }

            // Update item
            $data = [
                'product_name' => $product_name,
                'generic_name' => $generic_name,
                'ndc' => $ndc,
                'sku' => $sku,
                'dosage_strength' => $dosage_strength,
                'form' => $form,
                'manufacturer' => $supplier['name'],
                'unit_of_measure' => $unit_of_measure,
                'category' => $category,
                'storage_requirements' => $storage_requirements,
                'quantity' => $quantity,
                'reorder_point' => $reorder_point,
                'max_stock_level' => $max_stock_level,
                'expiry_date' => $expiry_date,
                'lot_number' => $lot_number,
                'product_image' => $product_image,
                'description' => $description,
                'unit_price' => $unit_price
            ];

            $set_clause = implode(' = ?, ', array_keys($data)) . ' = ?';
            $sql = "UPDATE inventory SET $set_clause WHERE id = ?";
            $params = array_values($data);
            $params[] = $id;

            if (updateData($sql, $params)) {
                $pdo->commit();
                $_SESSION['success_message'] = "Item updated successfully.";
                header('Location: ' . getUrlPath('pages/inventory/list.php'));
                exit();
            } else {
                throw new Exception("Failed to update item");
            }
        } catch (Exception $e) {
            $pdo->rollBack();
            $errors[] = "Error updating item: " . $e->getMessage();
        }
    }
}

// Get unique categories for dropdown
$categories = getRows("SELECT DISTINCT category FROM inventory WHERE category IS NOT NULL ORDER BY category");

// Include header after all potential redirects
require_once getIncludePath('includes/header.php');
?>

<div class="container-fluid">
    <div class="row justify-content-center">
        <div class="col-md-8">
            <div class="card">
                <div class="card-header">
                    <h3 class="card-title mb-0">Edit Item</h3>
                </div>
                <div class="card-body">
                    <?php if (!empty($errors)): ?>
                        <div class="alert alert-danger">
                            <ul class="mb-0">
                                <?php foreach ($errors as $error): ?>
                                    <li><?php echo $error; ?></li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                    <?php endif; ?>

                    <form method="POST" enctype="multipart/form-data">
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label for="product_name" class="form-label">Product Name*</label>
                                <input type="text" class="form-control" id="product_name" name="product_name" value="<?php echo htmlspecialchars($item['product_name']); ?>" required>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label for="generic_name" class="form-label">Generic Name</label>
                                <input type="text" class="form-control" id="generic_name" name="generic_name" value="<?php echo htmlspecialchars($item['generic_name']); ?>">
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label for="ndc" class="form-label">NDC</label>
                                <input type="text" class="form-control" id="ndc" name="ndc" value="<?php echo htmlspecialchars($item['ndc']); ?>">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label for="sku" class="form-label">SKU*</label>
                                <input type="text" class="form-control" id="sku" name="sku" value="<?php echo htmlspecialchars($item['sku']); ?>" required>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label for="dosage_strength" class="form-label">Dosage/Strength</label>
                                <input type="text" class="form-control" id="dosage_strength" name="dosage_strength" value="<?php echo htmlspecialchars($item['dosage_strength']); ?>">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label for="form" class="form-label">Form</label>
                                <input type="text" class="form-control" id="form" name="form" value="<?php echo htmlspecialchars($item['form']); ?>">
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label for="supplier_id" class="form-label">Supplier*</label>
                                <select class="form-control" id="supplier_id" name="supplier_id" required>
                                    <option value="">Select Supplier</option>
                                    <?php foreach ($suppliers as $supplier): ?>
                                        <option value="<?php echo $supplier['id']; ?>" <?php echo $current_supplier_id == $supplier['id'] ? 'selected' : ''; ?>>
                                            <?php echo htmlspecialchars($supplier['name']); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label for="unit_of_measure" class="form-label">Unit of Measure</label>
                                <input type="text" class="form-control" id="unit_of_measure" name="unit_of_measure" value="<?php echo htmlspecialchars($item['unit_of_measure']); ?>">
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label for="category" class="form-label">Category/Classification</label>
                                <input type="text" class="form-control" id="category" name="category" value="<?php echo htmlspecialchars($item['category']); ?>" list="categories">
                                <datalist id="categories">
                                    <?php foreach ($categories as $cat): ?>
                                        <option value="<?php echo htmlspecialchars($cat['category']); ?>">
                                    <?php endforeach; ?>
                                </datalist>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label for="storage_requirements" class="form-label">Storage Requirements</label>
                                <input type="text" class="form-control" id="storage_requirements" name="storage_requirements" value="<?php echo htmlspecialchars($item['storage_requirements']); ?>">
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-4 mb-3">
                                <label for="quantity" class="form-label">Quantity on Hand*</label>
                                <input type="number" class="form-control" id="quantity" name="quantity" min="0" value="<?php echo $item['quantity']; ?>" required>
                            </div>
                            <div class="col-md-4 mb-3">
                                <label for="reorder_point" class="form-label">Reorder Point</label>
                                <input type="number" class="form-control" id="reorder_point" name="reorder_point" min="0" value="<?php echo $item['reorder_point']; ?>">
                            </div>
                            <div class="col-md-4 mb-3">
                                <label for="max_stock_level" class="form-label">Maximum Stock Level</label>
                                <input type="number" class="form-control" id="max_stock_level" name="max_stock_level" min="0" value="<?php echo $item['max_stock_level']; ?>">
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label for="expiry_date" class="form-label">Expiration Date</label>
                                <input type="date" class="form-control" id="expiry_date" name="expiry_date" value="<?php echo $item['expiry_date']; ?>">
                            </div>
                            <div class="col-md-6 mb-3">
                                <label for="lot_number" class="form-label">Lot Number</label>
                                <input type="text" class="form-control" id="lot_number" name="lot_number" value="<?php echo htmlspecialchars($item['lot_number']); ?>">
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label for="unit_price" class="form-label">Unit Price*</label>
                                <input type="number" class="form-control" id="unit_price" name="unit_price" min="0" step="0.01" value="<?php echo $item['unit_price']; ?>" required>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label for="product_image" class="form-label">Product Image</label>
                                <?php if ($item['product_image']): ?>
                                    <div class="mb-2">
                                        <img src="<?php echo getUrlPath($item['product_image']); ?>" alt="Current Product Image" style="max-width: 100px; height: auto;">
                                    </div>
                                <?php endif; ?>
                                <input type="file" class="form-control" id="product_image" name="product_image" accept="image/*">
                            </div>
                        </div>

                        <div class="mb-3">
                            <label for="description" class="form-label">Description</label>
                            <textarea class="form-control" id="description" name="description" rows="3"><?php echo htmlspecialchars($item['description']); ?></textarea>
                        </div>

                        <div class="d-grid gap-2">
                            <button type="submit" class="btn btn-primary">Update Item</button>
                            <a href="<?php echo getUrlPath('pages/inventory/list.php'); ?>" class="btn btn-secondary">Cancel</a>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>

<?php require_once getIncludePath('includes/footer.php'); ?> 