<?php
session_start();
require_once dirname(__DIR__, 2) . DIRECTORY_SEPARATOR . 'config' . DIRECTORY_SEPARATOR . 'config.php';
require_once getIncludePath('includes/header.php');

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: ' . getUrlPath('pages/login.php'));
    exit();
}

// Get database connection
$pdo = getDBConnection();

// Search and filter parameters
$search = isset($_GET['search']) ? $_GET['search'] : '';
$category = isset($_GET['category']) ? $_GET['category'] : '';
$manufacturer = isset($_GET['manufacturer']) ? $_GET['manufacturer'] : '';
$stock_status = isset($_GET['stock_status']) ? $_GET['stock_status'] : 'all';
$expiry_filter = isset($_GET['expiry_filter']) ? $_GET['expiry_filter'] : 'all';
$sort = isset($_GET['sort']) ? $_GET['sort'] : 'product_name';
$order = isset($_GET['order']) ? $_GET['order'] : 'ASC';

// Build query
$query = "SELECT * FROM inventory WHERE 1=1";
$params = [];

if ($search) {
    $query .= " AND (product_name LIKE ? OR generic_name LIKE ? OR sku LIKE ? OR ndc LIKE ?)";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $params[] = "%$search%";
    $params[] = "%$search%";
}

if ($category) {
    $query .= " AND category = ?";
    $params[] = $category;
}

if ($manufacturer) {
    $query .= " AND manufacturer = ?";
    $params[] = $manufacturer;
}

// Apply stock status filter
switch ($stock_status) {
    case 'low':
        $query .= " AND quantity <= reorder_point";
        break;
    case 'out':
        $query .= " AND quantity = 0";
        break;
}

// Apply expiry date filter
switch ($expiry_filter) {
    case '30days':
        $query .= " AND expiry_date BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 30 DAY)";
        break;
    case '90days':
        $query .= " AND expiry_date BETWEEN CURDATE() AND DATE_ADD(CURDATE(), INTERVAL 90 DAY)";
        break;
    case 'expired':
        $query .= " AND expiry_date < CURDATE()";
        break;
}

// Apply sorting
switch ($sort) {
    case 'expiry_date':
        $query .= " ORDER BY CASE 
            WHEN expiry_date IS NULL THEN '9999-12-31'
            ELSE expiry_date 
        END $order";
        break;
    case 'quantity':
        $query .= " ORDER BY quantity $order";
        break;
    default:
        $query .= " ORDER BY $sort $order";
}

// Debug information
echo "<!-- Debug Info:
Query: " . htmlspecialchars($query) . "
Params: " . print_r($params, true) . "
-->";

$items = getRows($query, $params);

// Debug information
echo "<!-- Number of items found: " . count($items) . " -->";

// Get unique categories and manufacturers for filters
$categories = getRows("SELECT DISTINCT category FROM inventory WHERE category IS NOT NULL ORDER BY category");
$manufacturers = getRows("SELECT DISTINCT manufacturer FROM inventory WHERE manufacturer IS NOT NULL ORDER BY manufacturer");

// Debug information
echo "<!-- Categories: " . print_r($categories, true) . " -->";
echo "<!-- Manufacturers: " . print_r($manufacturers, true) . " -->";
?>

<div class="container-fluid">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h2>Inventory List</h2>
        <div>
            <button type="button" class="btn btn-success me-2" data-bs-toggle="modal" data-bs-target="#scanModal">
                <i class="fas fa-barcode"></i> Scan Barcode
            </button>
            <a href="<?php echo getUrlPath('pages/inventory/adjustments.php'); ?>" class="btn btn-warning me-2">
                <i class="fas fa-balance-scale"></i> Adjustments
            </a>
            <a href="<?php echo getUrlPath('pages/inventory/add.php'); ?>" class="btn btn-primary">
                <i class="fas fa-plus"></i> Add New Item
            </a>
        </div>
    </div>

    <!-- Search and Filter Form -->
    <div class="card mb-4">
        <div class="card-body">
            <form method="GET" class="row g-3">
                <div class="col-md-3">
                    <input type="text" class="form-control" name="search" placeholder="Search by name, SKU, or NDC..." value="<?php echo htmlspecialchars($search); ?>">
                </div>
                <div class="col-md-2">
                    <select class="form-select" name="category">
                        <option value="">All Categories</option>
                        <?php foreach ($categories as $cat): ?>
                            <option value="<?php echo htmlspecialchars($cat['category']); ?>" <?php echo $category === $cat['category'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($cat['category']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="col-md-2">
                    <select class="form-select" name="manufacturer">
                        <option value="">All Manufacturers</option>
                        <?php foreach ($manufacturers as $man): ?>
                            <option value="<?php echo htmlspecialchars($man['manufacturer']); ?>" <?php echo $manufacturer === $man['manufacturer'] ? 'selected' : ''; ?>>
                                <?php echo htmlspecialchars($man['manufacturer']); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="col-md-2">
                    <select class="form-select" name="stock_status">
                        <option value="all" <?php echo $stock_status === 'all' ? 'selected' : ''; ?>>All Stock</option>
                        <option value="low" <?php echo $stock_status === 'low' ? 'selected' : ''; ?>>Low Stock</option>
                        <option value="out" <?php echo $stock_status === 'out' ? 'selected' : ''; ?>>Out of Stock</option>
                    </select>
                </div>
                <div class="col-md-2">
                    <select class="form-select" name="expiry_filter">
                        <option value="all" <?php echo $expiry_filter === 'all' ? 'selected' : ''; ?>>All Expiry</option>
                        <option value="30days" <?php echo $expiry_filter === '30days' ? 'selected' : ''; ?>>Expiring in 30 days</option>
                        <option value="90days" <?php echo $expiry_filter === '90days' ? 'selected' : ''; ?>>Expiring in 90 days</option>
                        <option value="expired" <?php echo $expiry_filter === 'expired' ? 'selected' : ''; ?>>Expired</option>
                    </select>
                </div>
                <div class="col-md-1">
                    <button type="submit" class="btn btn-primary w-100">Filter</button>
                </div>
            </form>
        </div>
    </div>

    <!-- Sort Options -->
    <div class="card mb-4">
        <div class="card-body">
            <div class="row g-2">
                <div class="col-md-2">
                    <select class="form-select" onchange="window.location.href=this.value">
                        <option value="?sort=product_name&order=ASC" <?php echo $sort === 'product_name' && $order === 'ASC' ? 'selected' : ''; ?>>Name (A-Z)</option>
                        <option value="?sort=product_name&order=DESC" <?php echo $sort === 'product_name' && $order === 'DESC' ? 'selected' : ''; ?>>Name (Z-A)</option>
                        <option value="?sort=expiry_date&order=ASC" <?php echo $sort === 'expiry_date' && $order === 'ASC' ? 'selected' : ''; ?>>Expiry (Nearest)</option>
                        <option value="?sort=expiry_date&order=DESC" <?php echo $sort === 'expiry_date' && $order === 'DESC' ? 'selected' : ''; ?>>Expiry (Farthest)</option>
                        <option value="?sort=quantity&order=DESC" <?php echo $sort === 'quantity' && $order === 'DESC' ? 'selected' : ''; ?>>Quantity (High-Low)</option>
                        <option value="?sort=quantity&order=ASC" <?php echo $sort === 'quantity' && $order === 'ASC' ? 'selected' : ''; ?>>Quantity (Low-High)</option>
                    </select>
                </div>
            </div>
        </div>
    </div>

    <!-- Inventory Table -->
    <div class="card">
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-striped">
                    <thead>
                        <tr>
                            <th>Image</th>
                            <th>Product Name</th>
                            <th>NDC</th>
                            <th>SKU</th>
                            <th>Quantity</th>
                            <th>Reorder Point</th>
                            <th>Expiry Date</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($items)): ?>
                            <tr>
                                <td colspan="8" class="text-center py-4">
                                    <div class="alert alert-info mb-0">
                                        No items found. Try adjusting your search or filters.
                                    </div>
                                </td>
                            </tr>
                        <?php else: ?>
                            <?php foreach ($items as $item): ?>
                            <tr>
                                <td>
                                    <?php if ($item['product_image']): ?>
                                        <img src="<?php echo getUrlPath($item['product_image']); ?>" alt="Product" style="width: 50px; height: 50px; object-fit: cover;">
                                    <?php else: ?>
                                        <div class="bg-light d-flex align-items-center justify-content-center" style="width: 50px; height: 50px;">
                                            <i class="fas fa-image text-muted"></i>
                                        </div>
                                    <?php endif; ?>
                                </td>
                                <td><?php echo htmlspecialchars($item['product_name']); ?></td>
                                <td><?php echo htmlspecialchars($item['ndc']); ?></td>
                                <td><?php echo htmlspecialchars($item['sku']); ?></td>
                                <td>
                                    <span class="badge <?php echo $item['quantity'] <= $item['reorder_point'] ? 'bg-danger' : 'bg-success'; ?>">
                                        <?php echo $item['quantity']; ?>
                                    </span>
                                    <?php if ($item['quantity'] <= $item['reorder_point']): ?>
                                        <small class="text-danger d-block">Reorder Point: <?php echo $item['reorder_point']; ?></small>
                                    <?php endif; ?>
                                </td>
                                <td><?php echo $item['reorder_point']; ?></td>
                                <td>
                                    <?php if ($item['expiry_date']): ?>
                                        <?php
                                        $expiry_date = strtotime($item['expiry_date']);
                                        $today = time();
                                        $days_until_expiry = floor(($expiry_date - $today) / (60 * 60 * 24));
                                        $expiry_class = $days_until_expiry <= 30 ? 'text-danger' : ($days_until_expiry <= 90 ? 'text-warning' : 'text-success');
                                        ?>
                                        <span class="<?php echo $expiry_class; ?>">
                                            <?php echo date('Y-m-d', $expiry_date); ?>
                                        </span>
                                    <?php else: ?>
                                        N/A
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <a href="<?php echo getUrlPath('pages/inventory/edit.php?id=' . $item['id']); ?>" class="btn btn-sm btn-warning">Edit</a>
                                    <a href="<?php echo getUrlPath('pages/inventory/delete.php?id=' . $item['id']); ?>" class="btn btn-sm btn-danger" onclick="return confirm('Are you sure you want to delete this item?')">Delete</a>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>

<!-- Barcode Scan Modal -->
<div class="modal fade" id="scanModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Scan Barcode</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <div class="mb-3">
                    <label for="barcode" class="form-label">Enter Barcode</label>
                    <input type="text" class="form-control" id="barcode" autofocus>
                </div>
                <div id="scanResult"></div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
            </div>
        </div>
    </div>
</div>

<script>
document.getElementById('barcode').addEventListener('keypress', function(e) {
    if (e.key === 'Enter') {
        const barcode = this.value;
        // Here you would typically make an AJAX call to search for the product
        // For now, we'll just redirect to the search
        window.location.href = 'list.php?search=' + encodeURIComponent(barcode);
    }
});
</script>

<?php require_once getIncludePath('includes/footer.php'); ?> 