<?php
session_start();
require_once dirname(__DIR__, 2) . DIRECTORY_SEPARATOR . 'config' . DIRECTORY_SEPARATOR . 'config.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: ' . getUrlPath('pages/login.php'));
    exit();
}

// Get database connection
$pdo = getDBConnection();

// Get suppliers for dropdown
$suppliers = getRows("SELECT * FROM suppliers ORDER BY name");

// Get products for dropdown
$products = getRows("SELECT * FROM inventory ORDER BY product_name");

$errors = [];
$success = false;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $supplier_id = $_POST['supplier_id'] ?? '';
    $receipt_date = $_POST['receipt_date'] ?? '';
    $products_data = $_POST['products'] ?? [];

    if (empty($supplier_id)) {
        $errors[] = "Please select a supplier";
    }

    if (empty($receipt_date)) {
        $errors[] = "Please select a receipt date";
    }

    if (empty($products_data)) {
        $errors[] = "Please add at least one product";
    }

    if (empty($errors)) {
        try {
            $pdo->beginTransaction();

            // Insert receiving record
            $sql = "INSERT INTO receiving_records (supplier_id, receipt_date, created_by) VALUES (?, ?, ?)";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([$supplier_id, $receipt_date, $_SESSION['user_id']]);
            $receiving_id = $pdo->lastInsertId();

            // Insert receiving items
            $sql = "INSERT INTO receiving_items (receiving_record_id, product_id, quantity, lot_number, expiration_date) 
                    VALUES (?, ?, ?, ?, ?)";
            $stmt = $pdo->prepare($sql);

            foreach ($products_data as $product) {
                $stmt->execute([
                    $receiving_id,
                    $product['product_id'],
                    $product['quantity'],
                    $product['lot_number'],
                    $product['expiration_date']
                ]);

                // Update inventory quantity
                $sql = "UPDATE inventory SET quantity = quantity + ? WHERE id = ?";
                $stmt = $pdo->prepare($sql);
                $stmt->execute([$product['quantity'], $product['product_id']]);
            }

            $pdo->commit();
            $success = true;
        } catch (Exception $e) {
            $pdo->rollBack();
            $errors[] = "Error processing receipt: " . $e->getMessage();
        }
    }
}

require_once getIncludePath('includes/header.php');
?>

<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <div class="card">
                <div class="card-header">
                    <h3 class="card-title">Record Receipt</h3>
                </div>
                <div class="card-body">
                    <?php if (!empty($errors)): ?>
                        <div class="alert alert-danger">
                            <ul class="mb-0">
                                <?php foreach ($errors as $error): ?>
                                    <li><?php echo htmlspecialchars($error); ?></li>
                                <?php endforeach; ?>
                            </ul>
                        </div>
                    <?php endif; ?>

                    <?php if ($success): ?>
                        <div class="alert alert-success">
                            Receipt recorded successfully!
                        </div>
                    <?php endif; ?>

                    <form method="POST" id="receivingForm">
                        <div class="row mb-4">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="supplier_id">Supplier</label>
                                    <select class="form-control" id="supplier_id" name="supplier_id" required>
                                        <option value="">Select Supplier</option>
                                        <?php foreach ($suppliers as $supplier): ?>
                                            <option value="<?php echo $supplier['id']; ?>">
                                                <?php echo htmlspecialchars($supplier['name']); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="receipt_date">Receipt Date</label>
                                    <input type="date" class="form-control" id="receipt_date" name="receipt_date" 
                                           value="<?php echo date('Y-m-d'); ?>" required>
                                </div>
                            </div>
                        </div>

                        <div id="productsContainer">
                            <div class="product-item mb-3">
                                <div class="row">
                                    <div class="col-md-4">
                                        <div class="form-group">
                                            <label>Product</label>
                                            <select class="form-control product-select" name="products[0][product_id]" required>
                                                <option value="">Select Product</option>
                                                <?php foreach ($products as $product): ?>
                                                    <option value="<?php echo $product['id']; ?>">
                                                        <?php echo htmlspecialchars($product['product_name']); ?>
                                                    </option>
                                                <?php endforeach; ?>
                                            </select>
                                        </div>
                                    </div>
                                    <div class="col-md-2">
                                        <div class="form-group">
                                            <label>Quantity</label>
                                            <input type="number" class="form-control" name="products[0][quantity]" 
                                                   min="1" required>
                                        </div>
                                    </div>
                                    <div class="col-md-2">
                                        <div class="form-group">
                                            <label>Lot Number</label>
                                            <input type="text" class="form-control" name="products[0][lot_number]" required>
                                        </div>
                                    </div>
                                    <div class="col-md-2">
                                        <div class="form-group">
                                            <label>Expiration Date</label>
                                            <input type="date" class="form-control" name="products[0][expiration_date]" required>
                                        </div>
                                    </div>
                                    <div class="col-md-2">
                                        <div class="form-group">
                                            <label>&nbsp;</label>
                                            <button type="button" class="btn btn-danger btn-block remove-product" style="display: none;">
                                                Remove
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-12">
                                <button type="button" class="btn btn-secondary" id="addProduct">Add Product</button>
                                <button type="submit" class="btn btn-primary">Save Receipt</button>
                                <button type="reset" class="btn btn-default">Clear Form</button>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const productsContainer = document.getElementById('productsContainer');
    const addProductBtn = document.getElementById('addProduct');
    let productCount = 1;

    addProductBtn.addEventListener('click', function() {
        const template = document.querySelector('.product-item').cloneNode(true);
        
        // Update names and IDs
        template.querySelectorAll('select, input').forEach(input => {
            input.name = input.name.replace('[0]', `[${productCount}]`);
            input.value = '';
        });

        // Show remove button
        template.querySelector('.remove-product').style.display = 'block';

        productsContainer.appendChild(template);
        productCount++;
    });

    // Remove product
    productsContainer.addEventListener('click', function(e) {
        if (e.target.classList.contains('remove-product')) {
            e.target.closest('.product-item').remove();
        }
    });
});
</script>

<?php require_once getIncludePath('includes/footer.php'); ?> 