<?php
session_start();
require_once dirname(__DIR__, 2) . DIRECTORY_SEPARATOR . 'config' . DIRECTORY_SEPARATOR . 'config.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: ' . getUrlPath('pages/login.php'));
    exit();
}

// Get database connection
$pdo = getDBConnection();

// Get categories for filters
$categories = getRows("SELECT DISTINCT category FROM inventory WHERE category IS NOT NULL ORDER BY category");

// Get report data based on selection
$report_type = $_GET['type'] ?? '';
$start_date = $_GET['start_date'] ?? date('Y-m-d', strtotime('-30 days'));
$end_date = $_GET['end_date'] ?? date('Y-m-d');
$category = $_GET['category'] ?? '';

$report_data = [];
$report_title = '';

switch ($report_type) {
    case 'low_stock':
        $report_title = 'Low Stock Report';
        $report_data = getRows("
            SELECT * FROM inventory 
            WHERE quantity <= reorder_point 
            ORDER BY quantity ASC
        ");
        break;

    case 'expiration':
        $report_title = 'Expiration Report';
        $report_data = getRows("
            SELECT * FROM inventory 
            WHERE expiry_date IS NOT NULL 
            ORDER BY expiry_date ASC
        ");
        break;

    case 'valuation':
        $report_title = 'Inventory Valuation Report';
        $report_data = getRows("
            SELECT 
                category,
                COUNT(*) as total_items,
                SUM(quantity) as total_quantity,
                SUM(quantity * unit_price) as total_value
            FROM inventory 
            GROUP BY category
            ORDER BY total_value DESC
        ");
        break;

    case 'transactions':
        $report_title = 'Transaction History Report';
        $report_data = getRows("
            SELECT 
                r.receipt_date as date,
                'Receiving' as type,
                s.name as supplier,
                ri.quantity,
                p.product_name,
                u.username as user
            FROM receiving_records r
            JOIN suppliers s ON r.supplier_id = s.id
            JOIN receiving_items ri ON r.id = ri.receiving_record_id
            JOIN inventory p ON ri.product_id = p.id
            JOIN users u ON r.created_by = u.id
            WHERE r.receipt_date BETWEEN ? AND ?
            UNION ALL
            SELECT 
                d.dispensing_date as date,
                'Dispensing' as type,
                d.patient_name as supplier,
                di.quantity,
                p.product_name,
                u.username as user
            FROM dispensing_records d
            JOIN dispensing_items di ON d.id = di.dispensing_record_id
            JOIN inventory p ON di.product_id = p.id
            JOIN users u ON d.dispensed_by = u.id
            WHERE d.dispensing_date BETWEEN ? AND ?
            ORDER BY date DESC
        ", [$start_date, $end_date, $start_date, $end_date]);
        break;
}

// Handle export
if (isset($_GET['export']) && !empty($report_data)) {
    $export_type = $_GET['export'];
    
    if ($export_type === 'excel') {
        header('Content-Type: application/vnd.ms-excel');
        header('Content-Disposition: attachment;filename="' . $report_type . '_report.xls"');
        header('Cache-Control: max-age=0');
    } elseif ($export_type === 'pdf') {
        // You'll need to implement PDF generation using a library like TCPDF or FPDF
        // For now, we'll just show a message
        $error = "PDF export is not yet implemented.";
    }
}

require_once getIncludePath('includes/header.php');
?>

<div class="container-fluid">
    <div class="row">
        <div class="col-12">
            <h1 class="h3 mb-4 text-gray-800">Reports</h1>
        </div>
    </div>

    <!-- Report Selection and Filters -->
    <div class="row">
        <div class="col-12 mb-4">
            <div class="card shadow">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-primary">Select Report</h6>
                </div>
                <div class="card-body">
                    <form method="GET" class="row">
                        <div class="col-md-4 mb-3">
                            <label for="type">Report Type</label>
                            <select class="form-control" id="type" name="type" required>
                                <option value="">Select Report Type</option>
                                <option value="low_stock" <?php echo $report_type === 'low_stock' ? 'selected' : ''; ?>>Low Stock Report</option>
                                <option value="expiration" <?php echo $report_type === 'expiration' ? 'selected' : ''; ?>>Expiration Report</option>
                                <option value="valuation" <?php echo $report_type === 'valuation' ? 'selected' : ''; ?>>Inventory Valuation Report</option>
                                <option value="transactions" <?php echo $report_type === 'transactions' ? 'selected' : ''; ?>>Transaction History Report</option>
                            </select>
                        </div>
                        <div class="col-md-4 mb-3">
                            <label for="category">Category</label>
                            <select class="form-control" id="category" name="category">
                                <option value="">All Categories</option>
                                <?php foreach ($categories as $cat): ?>
                                    <option value="<?php echo htmlspecialchars($cat['category']); ?>" 
                                            <?php echo $category === $cat['category'] ? 'selected' : ''; ?>>
                                        <?php echo htmlspecialchars($cat['category']); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-4 mb-3">
                            <label>&nbsp;</label>
                            <button type="submit" class="btn btn-primary btn-block">Generate Report</button>
                        </div>
                    </form>

                    <?php if ($report_type === 'transactions'): ?>
                    <div class="row mt-3">
                        <div class="col-md-4">
                            <label for="start_date">Start Date</label>
                            <input type="date" class="form-control" id="start_date" name="start_date" 
                                   value="<?php echo $start_date; ?>">
                        </div>
                        <div class="col-md-4">
                            <label for="end_date">End Date</label>
                            <input type="date" class="form-control" id="end_date" name="end_date" 
                                   value="<?php echo $end_date; ?>">
                        </div>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>

    <!-- Report Display -->
    <?php if (!empty($report_data)): ?>
    <div class="row">
        <div class="col-12">
            <div class="card shadow mb-4">
                <div class="card-header py-3 d-flex justify-content-between align-items-center">
                    <h6 class="m-0 font-weight-bold text-primary"><?php echo $report_title; ?></h6>
                    <div>
                        <a href="?type=<?php echo $report_type; ?>&export=excel" class="btn btn-success btn-sm">
                            <i class="fas fa-file-excel"></i> Export Excel
                        </a>
                        <a href="?type=<?php echo $report_type; ?>&export=pdf" class="btn btn-danger btn-sm">
                            <i class="fas fa-file-pdf"></i> Export PDF
                        </a>
                    </div>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-bordered">
                            <thead>
                                <tr>
                                    <?php
                                    $headers = [];
                                    switch ($report_type) {
                                        case 'low_stock':
                                            $headers = ['Product Name', 'Category', 'Current Stock', 'Reorder Point', 'Unit Price'];
                                            break;
                                        case 'expiration':
                                            $headers = ['Product Name', 'Category', 'Quantity', 'Expiration Date', 'Days Until Expiry'];
                                            break;
                                        case 'valuation':
                                            $headers = ['Category', 'Total Items', 'Total Quantity', 'Total Value'];
                                            break;
                                        case 'transactions':
                                            $headers = ['Date', 'Type', 'Product', 'Quantity', 'Supplier/Patient', 'User'];
                                            break;
                                    }
                                    foreach ($headers as $header):
                                    ?>
                                        <th><?php echo $header; ?></th>
                                    <?php endforeach; ?>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($report_data as $row): ?>
                                    <tr>
                                        <?php
                                        switch ($report_type) {
                                            case 'low_stock':
                                                echo "<td>" . htmlspecialchars($row['product_name']) . "</td>";
                                                echo "<td>" . htmlspecialchars($row['category']) . "</td>";
                                                echo "<td>" . $row['quantity'] . "</td>";
                                                echo "<td>" . $row['reorder_point'] . "</td>";
                                                echo "<td>$" . number_format($row['unit_price'], 2) . "</td>";
                                                break;
                                            case 'expiration':
                                                $days_until_expiry = floor((strtotime($row['expiry_date']) - time()) / (60 * 60 * 24));
                                                echo "<td>" . htmlspecialchars($row['product_name']) . "</td>";
                                                echo "<td>" . htmlspecialchars($row['category']) . "</td>";
                                                echo "<td>" . $row['quantity'] . "</td>";
                                                echo "<td>" . date('Y-m-d', strtotime($row['expiry_date'])) . "</td>";
                                                echo "<td>" . $days_until_expiry . " days</td>";
                                                break;
                                            case 'valuation':
                                                echo "<td>" . htmlspecialchars($row['category']) . "</td>";
                                                echo "<td>" . $row['total_items'] . "</td>";
                                                echo "<td>" . $row['total_quantity'] . "</td>";
                                                echo "<td>$" . number_format($row['total_value'], 2) . "</td>";
                                                break;
                                            case 'transactions':
                                                echo "<td>" . date('Y-m-d', strtotime($row['date'])) . "</td>";
                                                echo "<td><span class='badge badge-" . ($row['type'] === 'Receiving' ? 'success' : 'info') . "'>" . 
                                                     htmlspecialchars($row['type']) . "</span></td>";
                                                echo "<td>" . htmlspecialchars($row['product_name']) . "</td>";
                                                echo "<td>" . $row['quantity'] . "</td>";
                                                echo "<td>" . htmlspecialchars($row['supplier']) . "</td>";
                                                echo "<td>" . htmlspecialchars($row['user']) . "</td>";
                                                break;
                                        }
                                        ?>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <?php endif; ?>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Add date range fields to form when transactions report is selected
    const reportType = document.getElementById('type');
    const form = reportType.closest('form');
    
    reportType.addEventListener('change', function() {
        const dateRangeFields = document.querySelector('.row.mt-3');
        if (this.value === 'transactions') {
            if (!dateRangeFields) {
                const newRow = document.createElement('div');
                newRow.className = 'row mt-3';
                newRow.innerHTML = `
                    <div class="col-md-4">
                        <label for="start_date">Start Date</label>
                        <input type="date" class="form-control" id="start_date" name="start_date" 
                               value="<?php echo $start_date; ?>">
                    </div>
                    <div class="col-md-4">
                        <label for="end_date">End Date</label>
                        <input type="date" class="form-control" id="end_date" name="end_date" 
                               value="<?php echo $end_date; ?>">
                    </div>
                `;
                form.appendChild(newRow);
            }
        } else if (dateRangeFields) {
            dateRangeFields.remove();
        }
    });
});
</script>

<?php require_once getIncludePath('includes/footer.php'); ?> 